SET QUOTED_IDENTIFIER ON

GO

-- ============================================================================
-- FIX: Merge Duplicate Campaigns
-- ============================================================================
SET ANSI_NULLS ON;

-- ============================================================================
-- This script fixes duplicate campaigns that were created due to a bug in
-- TrackBilletsDataJob where cobbles processed late would break campaign
-- continuation logic for all campaigns in the batch.
--
-- The fix:
-- 1. Detects consecutive campaigns with same Section+Heat (no other production in between)
-- 2. Merges them into one campaign (earliest CampaignId becomes the master)
-- 3. Updates Billets foreign keys to point to the master campaign
-- 4. Deletes the duplicate campaigns
-- ============================================================================

SET NOCOUNT ON;

BEGIN TRANSACTION;

BEGIN TRY
    -- ============================================================================
    -- STEP 1: Identify groups of consecutive campaigns to merge
    -- ============================================================================
    -- A "group" is formed when consecutive campaigns (by Start time) have the
    -- same SectionId and HeatId. When the Section/Heat changes, a new group starts.

    ;WITH OrderedCampaigns AS (
        SELECT
            CampaignId,
            SectionId,
            HeatId,
            [Start],
            [End],
            CampaignNumber,
            EventId,
            ChargedBillets,
            ChargedWeightHot,
            ChargedWeightCold,
            RolledBillets,
            RolledWeightHot,
            RolledWeightCold,
            Cobbles,
            RejectedBillets,
            RejectedWeightHot,
            RejectedWeightCold,
            IsActive,
            ROW_NUMBER() OVER (ORDER BY [Start], CampaignId) AS RowNum,
            LAG(SectionId) OVER (ORDER BY [Start], CampaignId) AS PrevSectionId,
            LAG(HeatId) OVER (ORDER BY [Start], CampaignId) AS PrevHeatId
        FROM M2.Campaigns
    ),
    GroupStarts AS (
        -- Mark where a new group starts (when Section/Heat changes or first row)
        SELECT
            *,
            CASE
                WHEN PrevSectionId IS NULL THEN 1  -- First row
                WHEN SectionId <> PrevSectionId OR HeatId <> PrevHeatId THEN 1  -- Change detected
                ELSE 0
            END AS IsGroupStart
        FROM OrderedCampaigns
    ),
    GroupIds AS (
        -- Assign a GroupId by cumulative sum of IsGroupStart
        SELECT
            *,
            SUM(IsGroupStart) OVER (ORDER BY RowNum) AS GroupId
        FROM GroupStarts
    ),
    GroupsToMerge AS (
        -- Find groups that have more than 1 campaign (these need merging)
        SELECT GroupId
        FROM GroupIds
        GROUP BY GroupId
        HAVING COUNT(*) > 1
    ),
    MasterCampaigns AS (
        -- For each group to merge, select the earliest CampaignId as master
        SELECT
            g.GroupId,
            MIN(g.CampaignId) AS MasterCampaignId,
            MIN(g.[Start]) AS NewStart,
            MAX(g.[End]) AS NewEnd,
            SUM(g.ChargedBillets) AS TotalChargedBillets,
            SUM(g.ChargedWeightHot) AS TotalChargedWeightHot,
            SUM(g.ChargedWeightCold) AS TotalChargedWeightCold,
            SUM(g.RolledBillets) AS TotalRolledBillets,
            SUM(g.RolledWeightHot) AS TotalRolledWeightHot,
            SUM(g.RolledWeightCold) AS TotalRolledWeightCold,
            SUM(g.Cobbles) AS TotalCobbles,
            SUM(g.RejectedBillets) AS TotalRejectedBillets,
            SUM(g.RejectedWeightHot) AS TotalRejectedWeightHot,
            SUM(g.RejectedWeightCold) AS TotalRejectedWeightCold
        FROM GroupIds g
        INNER JOIN GroupsToMerge gtm ON g.GroupId = gtm.GroupId
        GROUP BY g.GroupId
    )
    SELECT
        g.GroupId,
        g.CampaignId,
        g.SectionId,
        g.HeatId,
        g.[Start],
        g.[End],
        m.MasterCampaignId,
        CASE WHEN g.CampaignId = m.MasterCampaignId THEN 'MASTER' ELSE 'DUPLICATE' END AS Role
    INTO #CampaignsToProcess
    FROM GroupIds g
    INNER JOIN MasterCampaigns m ON g.GroupId = m.GroupId;

    -- Show what will be processed
    PRINT '=== CAMPAIGNS TO BE MERGED ===';
    SELECT
        GroupId,
        CampaignId,
        SectionId,
        HeatId,
        [Start],
        [End],
        MasterCampaignId,
        Role
    FROM #CampaignsToProcess
    ORDER BY GroupId, [Start];

    -- Count summary
    DECLARE @GroupCount INT = (SELECT COUNT(DISTINCT GroupId) FROM #CampaignsToProcess);
    DECLARE @DuplicateCount INT = (SELECT COUNT(*) FROM #CampaignsToProcess WHERE Role = 'DUPLICATE');

    PRINT '';
    PRINT 'Groups to merge: ' + CAST(@GroupCount AS VARCHAR(10));
    PRINT 'Duplicate campaigns to delete: ' + CAST(@DuplicateCount AS VARCHAR(10));

    -- ============================================================================
    -- STEP 2: Update master campaigns with aggregated values
    -- ============================================================================
    PRINT '';
    PRINT '=== UPDATING MASTER CAMPAIGNS ===';

    ;WITH AggregatedData AS (
        SELECT
            MasterCampaignId,
            MIN(p.[Start]) AS NewStart,
            MAX(p.[End]) AS NewEnd,
            SUM(c.ChargedBillets) AS TotalChargedBillets,
            SUM(c.ChargedWeightHot) AS TotalChargedWeightHot,
            SUM(c.ChargedWeightCold) AS TotalChargedWeightCold,
            SUM(c.RolledBillets) AS TotalRolledBillets,
            SUM(c.RolledWeightHot) AS TotalRolledWeightHot,
            SUM(c.RolledWeightCold) AS TotalRolledWeightCold,
            SUM(c.Cobbles) AS TotalCobbles,
            SUM(c.RejectedBillets) AS TotalRejectedBillets,
            SUM(c.RejectedWeightHot) AS TotalRejectedWeightHot,
            SUM(c.RejectedWeightCold) AS TotalRejectedWeightCold
        FROM #CampaignsToProcess p
        INNER JOIN M2.Campaigns c ON p.CampaignId = c.CampaignId
        GROUP BY MasterCampaignId
    )
    UPDATE c
    SET
        c.[Start] = a.NewStart,
        c.[End] = a.NewEnd,
        c.ChargedBillets = a.TotalChargedBillets,
        c.ChargedWeightHot = a.TotalChargedWeightHot,
        c.ChargedWeightCold = a.TotalChargedWeightCold,
        c.RolledBillets = a.TotalRolledBillets,
        c.RolledWeightHot = a.TotalRolledWeightHot,
        c.RolledWeightCold = a.TotalRolledWeightCold,
        c.Cobbles = a.TotalCobbles,
        c.RejectedBillets = a.TotalRejectedBillets,
        c.RejectedWeightHot = a.TotalRejectedWeightHot,
        c.RejectedWeightCold = a.TotalRejectedWeightCold
    FROM M2.Campaigns c
    INNER JOIN AggregatedData a ON c.CampaignId = a.MasterCampaignId;

    PRINT 'Master campaigns updated: ' + CAST(@@ROWCOUNT AS VARCHAR(10));

    -- ============================================================================
    -- STEP 3: Update Billets to point to master campaigns
    -- ============================================================================
    PRINT '';
    PRINT '=== UPDATING BILLETS FOREIGN KEYS ===';

    UPDATE b
    SET b.CampaignId = p.MasterCampaignId
    FROM M2.Billets b
    INNER JOIN #CampaignsToProcess p ON b.CampaignId = p.CampaignId
    WHERE p.Role = 'DUPLICATE';

    PRINT 'Billets updated: ' + CAST(@@ROWCOUNT AS VARCHAR(10));

    -- ============================================================================
    -- STEP 4: Delete duplicate campaigns
    -- ============================================================================
    PRINT '';
    PRINT '=== DELETING DUPLICATE CAMPAIGNS ===';

    DELETE c
    FROM M2.Campaigns c
    INNER JOIN #CampaignsToProcess p ON c.CampaignId = p.CampaignId
    WHERE p.Role = 'DUPLICATE';

    PRINT 'Duplicate campaigns deleted: ' + CAST(@@ROWCOUNT AS VARCHAR(10));

    -- Cleanup
    DROP TABLE #CampaignsToProcess;

    -- ============================================================================
    -- VERIFICATION
    -- ============================================================================
    PRINT '';
    PRINT '=== VERIFICATION: Remaining consecutive duplicates ===';

    ;WITH OrderedCampaigns AS (
        SELECT
            CampaignId,
            SectionId,
            HeatId,
            [Start],
            LAG(SectionId) OVER (ORDER BY [Start], CampaignId) AS PrevSectionId,
            LAG(HeatId) OVER (ORDER BY [Start], CampaignId) AS PrevHeatId,
            LAG(CampaignId) OVER (ORDER BY [Start], CampaignId) AS PrevCampaignId
        FROM M2.Campaigns
    )
    SELECT *
    FROM OrderedCampaigns
    WHERE SectionId = PrevSectionId AND HeatId = PrevHeatId;

    IF @@ROWCOUNT = 0
        PRINT 'SUCCESS: No consecutive duplicate campaigns found!';
    ELSE
        PRINT 'WARNING: Some consecutive duplicates still remain!';

    COMMIT TRANSACTION;
    PRINT '';
    PRINT '=== TRANSACTION COMMITTED SUCCESSFULLY ===';

END TRY
BEGIN CATCH
    ROLLBACK TRANSACTION;

    PRINT '';
    PRINT '=== ERROR - TRANSACTION ROLLED BACK ===';
    PRINT 'Error Number: ' + CAST(ERROR_NUMBER() AS VARCHAR(10));
    PRINT 'Error Message: ' + ERROR_MESSAGE();
    PRINT 'Error Line: ' + CAST(ERROR_LINE() AS VARCHAR(10));

    -- Cleanup temp table if exists
    IF OBJECT_ID('tempdb..#CampaignsToProcess') IS NOT NULL
        DROP TABLE #CampaignsToProcess;
END CATCH;

GO

-- Re-enable TrackBilletsMainJob after deployment
UPDATE [SYSTEM].catJobs SET Enabled = 1 WHERE sProcedure = 'M2.TrackBilletsMainJob';
PRINT 'TrackBilletsMainJob re-enabled after deployment';
GO
