SET QUOTED_IDENTIFIER ON

GO

-- INSERT NEW catSettings
INSERT INTO UTL.catSettings (SettingName, Value, Description)
SELECT SettingName, Value, Description
FROM (VALUES
    ('SCHEDULING.MAX_CASCADE_ITERATIONS', 100, 'Maximum iterations for cascade collision resolution'),
    ('SCHEDULING.MAX_FIXED_RESOLUTION_LOOPS', 15, 'Maximum Fixed events to process per Production event'),
    ('SCHEDULING.MAX_PREPROCESS_ITERATIONS', 50, 'Maximum iterations for new event pre-processing'),
    ('SCHEDULING.MAX_COMPACTION_ITERATIONS', 100, 'Maximum iterations for schedule compaction')
) AS NewSettings(SettingName, Value, Description)
WHERE NOT EXISTS (
    SELECT 1 FROM UTL.catSettings 
    WHERE catSettings.SettingName = NewSettings.SettingName
)

-- ================================================================================================
-- PRODUCTION EVENTS IMPORT SCRIPT
-- Imports Excel data from September, October, November, December 2025 and January 2026
-- ================================================================================================

SET NOCOUNT ON;
GO

DECLARE @User VARCHAR(50) = 'Admin';
DECLARE @Year INT = 2025;
DECLARE @CutoffDate DATETIME = '2025-12-02 00:00:00'; 

-- Temporary table to hold Excel data
IF OBJECT_ID('tempdb..#ExcelData') IS NOT NULL DROP TABLE #ExcelData;
CREATE TABLE #ExcelData (
    RowNum INT IDENTITY(1,1),
    MonthName VARCHAR(20),
    StartDay INT,
	InquiryStatusCode VARCHAR(2),
	Hour INT,
    SectionName VARCHAR(50),
    ProductDescription VARCHAR(MAX),
    ChargeTons REAL,
    RollTons REAL,
    FinTons REAL,
    Tph REAL,
    Hours REAL,
    YieldValue REAL,
    PlannedTons REAL,
    PlannedTph REAL,
    PlannedYield REAL
);

-- ================================================================================================
-- STEP 1: INSERT YOUR EXCEL DATA HERE
-- Copy data from Excel columns: A, B, F, G, H, I, J, K, L, Q, R, S
-- ================================================================================================

-- SEPTEMBER 2025 DATA
INSERT INTO #ExcelData (MonthName, StartDay, InquiryStatusCode, SectionName, ProductDescription, ChargeTons, RollTons, FinTons, Tph, Hours, YieldValue, PlannedTons, PlannedTph, PlannedYield)
VALUES 
	('September', 2, 'PI', '2658', '3" I BEAM @ 5.7', 280, 280, 204, 56.9, 5, 98.3, 235, 30.2, 87.7),
	('September', 2, 'PI', '2663', '3" S @ 7.5', 736, 640, 625, 26.7, 24, 87.0, 610, 22.5, 86.9),
	('September', 3, 'C', '2695', '4" I BEAM @ 3.25', 4810, 4294, 4271, 24.5, 175, 89.7, 4000, 26.0, 89.9),
	('September', 10, 'C', '2754', '4" I BEAM @ 3.49', 370, 338, 326, 28.1, 12, 91.4, 300, 25.1, 91.6),
	('September', 11, 'C', '2680', '4" S I BEAM @ 7.7', 2689, 2346, 2311, 36.1, 65, 87.3, 2300, 32.1, 89.8),
	('September', 14, 'C', '3137', '5" S- I Beam @ 7.2', 1394, 1258, 1201, 26.2, 48, 90.2, 1080, 28.0, 90.0),
	('September', 16, 'C', '2935', '5" S I BEAM @ 10.0', 772, 674, 650, 33.7, 20, 87.4, 625, 33.9, 86.0),
	('September', 17, 'C', '2759', '6" WF @ 8.5', 814, 632, 581, 16.2, 39, 77.6, 2000, 34.0, 89.2),
	('September', 18, 'C', '2725', '6" WF @ 9.0', 1064, 874, 780, 27.3, 32, 82.2, NULL, NULL, NULL),
	('September', 20, 'C', '3102', '6" WF @ 10.48', 4029, 3437, 3317, 29.6, 116, 85.3, 2000, 29.0, 87.3),
	('September', 25, 'C', '2734', '6" WF @ 12.0', 4199, 3757, 3736, 37.9, 99, 89.5, 2000, 34.0, 84.7),
	('September', 29, 'C', '2926', '6" WF @ 16.0', 457, 393, 387, 28.1, 14, 86.1, NULL, NULL, NULL),
	('September', 30, 'I', '2919', '3" I BEAM @ 2.58', 1001, 865, 803, 19.22, 45, 86.4, 1000, 18.0, 90.4);

-- OCTOBER 2025 DATA
INSERT INTO #ExcelData (MonthName, StartDay, InquiryStatusCode, SectionName, ProductDescription, ChargeTons, RollTons, FinTons, Tph, Hours, YieldValue, PlannedTons, PlannedTph, PlannedYield)
VALUES
	('October', 2, 'I', '2718', '3" I BEAM @ 2.90', 355, 323, 323, 23.1, 14, 90.9, 300, 20.9, 90.0),
	('October', 2, 'I', '2990', '3" I BEAM @ 3.78', 350, 308, 298, 12.8, 24, 87.9, 300, 17.9, 88.8),
	('October', 4, 'C', '2695', '4" I BEAM @ 3.25', 4615, 4159, 4092, 20.4, 204, 90.1, 4000, 26.0, 89.9),
	('October', 12, 'C', '2720', '4" I BEAM @ 3.46', 403, 332, 319, 22.1, 15, 82.3, 300, 24.0, 89.2),
	('October', 13, 'PI', '2744', '4" I BEAM @ 3.46', 1199, 1079, 1071, 18.6, 58, 90.0, 1000, 22.4, 90.2),
	('October', 15, 'PI', '2759', '6" WF @ 8.5', 2350, 1995, 1806, 23.5, 85, 84.9, 1905, 29.6, 84.7),
	('October', 18, 'PI', '2725', '6" WF @ 9.0', 3973, 3399, 3313, 31.5, 108, 85.6, 3300, 29.0, 87.3),
	('October', 24, 'C', '3102', '6" WF @ 10.48', 5997, 5115, 5066, 36.0, 142, 85.3, 4185, 33.1, 81.1),
	('October', 30, 'C', '2725', '6" WF @ 9.0', 50, 31, 31, 15.5, 2, 62.0, NULL, NULL, NULL),
	('October', 31, 'C', '2734', '6" WF @ 12.0', 4857, 4282, 4260, 31.6, 135, 88.2, 2305, 34.0, 89.2);

-- NOVEMBER 2025 DATA
INSERT INTO #ExcelData (MonthName, StartDay, InquiryStatusCode, SectionName, ProductDescription, ChargeTons, RollTons, FinTons, Tph, Hours, YieldValue, PlannedTons, PlannedTph, PlannedYield)
VALUES
	('November', 3, 'C', '2926', '6" WF @ 16.0', 1752, 1533, 1484, 33.3, 46, 87.5, NULL, NULL, NULL),
	('November', 4, 'C', '2658', '3" I BEAM @ 5.7', 2583, 2247, 2209, 23.6, 95, 87.0, NULL, NULL, NULL),
	('November', 8, 'C', '2663', '3" S @ 7.5', 710, 600, 567, 25, 24, 84.5, NULL, NULL, NULL),
	('November', 10, 'C', '2695', '4" I BEAM @ 3.25', 4891, 4264, 4120, 18.5, 230, 87.2, 4000, 26.0, 89.9),
	('November', 19, 'C', '2754', '4" I BEAM @ 3.49', 353, 320, 316, 24.6, 13, 90.7, 300, 25.1, 91.6),
	('November', 20, 'C', '2559', '4" I BEAM @ 4.08', 177, 157, 154, 19.6, 8, 88.6, 150, 26.8, 93.2),
	('November', 21, 'I', '2919', '3" I BEAM @ 2.58', 1121, 971, 934, 12.1, 80, 86.6, 1000, 18.0, 90.4),
	('November', 24, 'PS', '2680', '4" S I BEAM @ 7.7', 2867, 2507, 2436, 24.4, 100, 87.4, 2500, 32.1, 89.8);

-- DECEMBER 2025 DATA
INSERT INTO #ExcelData (MonthName, StartDay, InquiryStatusCode, SectionName, ProductDescription, ChargeTons, RollTons, FinTons, Tph, Hours, YieldValue, PlannedTons, PlannedTph, PlannedYield)
VALUES
	('December', 1, 'PS', '2934', '4" S @ 9.5', 121, 103, 98, 34.3, 3, 85.2, NULL, NULL, NULL),
	('December', 2, 'C', '2759', '6" WF @ 8.5', 542, 457, 450, 17.6, 26, 84.3, NULL, NULL, NULL),
	('December', 3, 'C', '2725', '6" WF @ 9.0', 4861, 4206, 4078, 27.8, 151, 86.5, NULL, NULL, NULL),
	('December', 10, 'C', '3102', '6" WF @ 10.48', 2042, 1655, 1655, 33.1, 50, 81.1, NULL, NULL, NULL),
	('December', 13, 'C', '2734', '6" WF @ 12.0', 4680, 4172, 4090, 34.0, 123, 89.2, NULL, NULL, NULL),
	('December', 18, 'C', '2722', '5" I BEAM @ 4.11 (5411-MAR)', 769, 615, 600, 10.9, 56, 80.0, NULL, NULL, NULL),
	('December', 21, 'PI', '2922', '6" CHANNEL @ 8.2', 1747, 1551, 1500, 29.1, 53, 88.8, NULL, NULL, NULL),
	('December', 23, 'PI', '2995', '6" CHANNEL @ 6.5', 789, 622, 600, 17.5, 36, 78.9, NULL, NULL, NULL),
	('December', 26, 'PI', '2921', '4" WF @ 13.0', 1762, 1533, 1500, 25, 61, 87.0, NULL, NULL, NULL),
	('December', 28, 'C', '2695', '4" I BEAM @ 3.25', 4624, 4158, 4000, 25.0, 160, 89.9, NULL, NULL, NULL);

-- JANUARY 2026 DATA
INSERT INTO #ExcelData (MonthName, StartDay, InquiryStatusCode, SectionName, ProductDescription, ChargeTons, RollTons, FinTons, Tph, Hours, YieldValue, PlannedTons, PlannedTph, PlannedYield)
VALUES
	('January', 4, 'PI', '2759', '6" WF @ 8.5', 2391, 2025, 2000, 29.6, 68, 84.7, NULL, NULL, NULL),
	('January', 7, 'PI', '2725', '6" WF @ 9.0', 2365, 2064, 2000, 29, 71, 87.3, NULL, NULL, NULL),
	('January', 10, 'PI', '3102', '6" WF @ 10.48', 2467, 2000, 2000, 33.1, 60, 81.1, NULL, NULL, NULL),
	('January', 13, 'PI', '2734', '6" WF @ 12.0', 2289, 2040, 2000, 34.0, 60, 89.2, NULL, NULL, NULL),
	('January', 16, 'PS', '2658', '3" I BEAM @ 5.7', 2254, 1976, 1900, 30.2, 65, 87.7, NULL, NULL, NULL),
	('January', 19, 'PS', '2663', '3" S @ 7.5', 270, 235, 224, 22.5, 10, 86.9, NULL, NULL, NULL),
	('January', 19, 'C', '2919', '3" I BEAM @ 2.58', 1191, 1076, 1000, 18, 60, 90.4, NULL, NULL, NULL),
	('January', 22, 'C', '2718', '3" I BEAM @ 2.90', 593, 533, 500, 20.9, 26, 90.0, NULL, NULL, NULL),
	('January', 23, 'C', '2990', '3" I BEAM @ 3.78', 610, 542, 500, 17.9, 30, 88.8, NULL, NULL, NULL),
	('January', 24, 'C', '2695', '4" I BEAM @ 3.25', 4624, 4157, 4000, 26.0, 160, 89.9, NULL, NULL, NULL),
	('January', 31, 'C', '2754', '4" I BEAM @ 3.49', 339, 310, 300, 25.1, 12, 91.6, NULL, NULL, NULL),
	('January', 31, 'PI', '2759', '6" WF @ 8.5', 209, 177, 175, 29.6, 6, 84.7, NULL, NULL, NULL);

-- ================================================================================================
-- STEP 2: PROCESS AND INSERT DATA
-- ================================================================================================

-- Working table with calculated fields
IF OBJECT_ID('tempdb..#ProcessedData') IS NOT NULL DROP TABLE #ProcessedData;
CREATE TABLE #ProcessedData (
    RowNum INT,
    StartDateTime DATETIME,
    EndDateTime DATETIME,
    SectionId INT,
    StatusCode CHAR(1),
    ProductDescription VARCHAR(MAX),
    -- Actual fields (for completed events)
    ActualChargedTons REAL,
    ActualRolledTons REAL,
    ActualFinishedTons REAL,
    ActualTPH REAL,
    ActualRollingHours REAL,
    ActualMillYield REAL,
    ActualFinYield REAL,
    -- Scheduled/Planned fields (always populated)
    ScheduledChargedTons REAL,
    ScheduledRolledTons REAL,
    ScheduledFinishedTons REAL,
    ScheduledTPH REAL,
    ScheduledRollingHours REAL,
	ScheduledMillYield REAL,
    ScheduledFinYield REAL,
    -- Planned fields
    PlannedTons REAL,
    PlannedTPH REAL,
    PlannedMillYield REAL,
    PlannedFinYield REAL,

	InquiryStatusCode VARCHAR(2)
);

-- Initial insert with default StartDateTime (first day of each month at 00:00)
INSERT INTO #ProcessedData (
    RowNum,
    StartDateTime,
    EndDateTime,
    SectionId,
    StatusCode,
    ProductDescription,
    ActualChargedTons,
    ActualRolledTons,
    ActualFinishedTons,
    ActualTPH,
    ActualRollingHours,
    ActualMillYield,
    ActualFinYield,
    ScheduledChargedTons,
    ScheduledRolledTons,
    ScheduledFinishedTons,
    ScheduledTPH,
    ScheduledRollingHours,
	ScheduledMillYield,
    ScheduledFinYield,
    PlannedTons,
    PlannedTPH,
    PlannedMillYield,
    PlannedFinYield,
	InquiryStatusCode
)
SELECT 
    ed.RowNum,
    -- Initial StartDateTime: first day of each month at 00:00 (will be updated later)
    UTL.ToUTC(DATETIMEFROMPARTS(
        CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END,
        CASE ed.MonthName
            WHEN 'September' THEN 9
            WHEN 'October' THEN 10
            WHEN 'November' THEN 11
            WHEN 'December' THEN 12
            WHEN 'January' THEN 1
        END,
        1,  -- Start at day 1
        0,  -- Hour 0
        0,  -- Minute 0
        0,  -- Second 0
        0   -- Millisecond 0
    )) AS StartDateTime,
    -- EndDateTime will be calculated later
    NULL AS EndDateTime,
    -- Lookup SectionId
    s.SectionId,
    -- Status Code based on cutoff date
    CASE
        WHEN DATETIMEFROMPARTS(
            CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END,
            CASE ed.MonthName
                WHEN 'September' THEN 9
                WHEN 'October' THEN 10
                WHEN 'November' THEN 11
                WHEN 'December' THEN 12
                WHEN 'January' THEN 1
            END,
            ed.StartDay,
            0, 0, 0, 0
        ) <= @CutoffDate
        THEN 'C'  -- Completed
        ELSE 'P'  -- Planned
    END AS StatusCode,
    ed.ProductDescription,
    -- Actual fields (only populated when StatusCode = 'C', i.e., <= CutoffDate)
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN ed.ChargeTons 
        ELSE -999 
    END AS ActualChargedTons,
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN ed.RollTons 
        ELSE -999 
    END AS ActualRolledTons,
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN ed.FinTons 
        ELSE -999 
    END AS ActualFinishedTons,
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN ed.Tph 
        ELSE -999 
    END AS ActualTPH,
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN ed.Hours 
        ELSE -999 
    END AS ActualRollingHours,
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN ed.YieldValue 
        ELSE -999 
    END AS ActualMillYield,
    -- ActualFinYield: FinTons / ChargeTons (only when completed)
    CASE 
        WHEN DATETIMEFROMPARTS(CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END, CASE ed.MonthName WHEN 'September' THEN 9 WHEN 'October' THEN 10 WHEN 'November' THEN 11 WHEN 'December' THEN 12 WHEN 'January' THEN 1 END, ed.StartDay, 0, 0, 0, 0) <= @CutoffDate 
        THEN CASE WHEN ed.ChargeTons > 0 THEN ed.FinTons / ed.ChargeTons * 100 ELSE -999 END
        ELSE -999 
    END AS ActualFinYield,
    -- Scheduled fields (ALWAYS populated, regardless of CutoffDate)
    ed.ChargeTons AS ScheduledChargedTons,
    ed.RollTons AS ScheduledRolledTons,
    ed.FinTons AS ScheduledFinishedTons,
    ISNULL(ed.Tph, -999) AS ScheduledTPH,
    ISNULL(ed.Hours, -999) AS ScheduledRollingHours,
    -- ScheduledFinYield: FinTons / ChargeTons (always calculated)
	isnull(ed.YieldValue, CASE WHEN ed.RollTons > 0 THEN ed.RollTons / ed.ChargeTons * 100 ELSE -999 END) AS ScheduledMillYield,
    CASE WHEN ed.ChargeTons > 0 THEN ed.FinTons / ed.ChargeTons * 100 ELSE -999 END AS ScheduledFinYield,

    -- Planned fields
    -- PlannedTons: if NULL, use FinTons
    ISNULL(ed.PlannedTons, ed.FinTons) AS PlannedTons,
    -- PlannedTPH: if NULL, use TonsPerHour from Sections table
    ISNULL(ed.PlannedTph, ISNULL(s.TonsPerHour, -999)) AS PlannedTPH,
    -- PlannedMillYield: if NULL, use MillYield from Sections table
    ISNULL(ed.PlannedYield, ISNULL(s.MillYield, -999)) AS PlannedMillYield,
    -- PlannedFinYield: get from Sections.FinishedYield
    ISNULL(s.FinishedYield, -999) AS PlannedFinYield,
	InquiryStatusCode
FROM #ExcelData ed
LEFT JOIN M2.Sections s ON s.SectionName = CAST(ed.SectionName AS VARCHAR(50))
WHERE ed.SectionName IS NOT NULL
ORDER BY ed.RowNum;

-- Update StartDateTime based on previous event's end time
-- Logic: Current StartDateTime = Previous StartDateTime + Previous Hours
-- We need to do this iteratively for each row
DECLARE @CurrentRowNum INT;
DECLARE @PrevStartDateTime DATETIME;
DECLARE @PrevHours REAL;
DECLARE @CurrentMonthName VARCHAR(50);
DECLARE @PrevMonthName VARCHAR(50);

-- Get the first row for each month and set it to the first day of that month at 00:00
UPDATE pd
SET StartDateTime = UTL.ToUTC(DATETIMEFROMPARTS(
    CASE ed.MonthName WHEN 'January' THEN @Year + 1 ELSE @Year END,
    CASE ed.MonthName
        WHEN 'September' THEN 9
        WHEN 'October' THEN 10
        WHEN 'November' THEN 11
        WHEN 'December' THEN 12
        WHEN 'January' THEN 1
    END,
    1,  -- First day of month
    0, 0, 0, 0
))
FROM #ProcessedData pd
INNER JOIN #ExcelData ed ON ed.RowNum = pd.RowNum
WHERE pd.RowNum = (
    SELECT MIN(pd2.RowNum)
    FROM #ProcessedData pd2
    INNER JOIN #ExcelData ed2 ON ed2.RowNum = pd2.RowNum
    WHERE ed2.MonthName = ed.MonthName
);

-- Now update all other rows based on the previous row's StartDateTime + Hours
-- Use a cursor-like approach to iterate through existing RowNums only
DECLARE @RowNumTable TABLE (RowNum INT);
INSERT INTO @RowNumTable (RowNum)
SELECT RowNum FROM #ProcessedData ORDER BY RowNum;

DECLARE @PrevRowNum INT = NULL;

DECLARE row_cursor CURSOR FOR
    SELECT RowNum FROM @RowNumTable ORDER BY RowNum;

OPEN row_cursor;
FETCH NEXT FROM row_cursor INTO @CurrentRowNum;

WHILE @@FETCH_STATUS = 0
BEGIN
    IF @PrevRowNum IS NOT NULL
    BEGIN
        -- Get previous row's data
        SELECT
            @PrevStartDateTime = pd.StartDateTime,
            @PrevHours = ed.Hours,
            @PrevMonthName = ed.MonthName
        FROM #ProcessedData pd
        INNER JOIN #ExcelData ed ON ed.RowNum = pd.RowNum
        WHERE pd.RowNum = @PrevRowNum;

        -- Update current row's StartDateTime (always based on previous event, no month reset)
        UPDATE #ProcessedData
        SET StartDateTime = DATEADD(HOUR, @PrevHours, @PrevStartDateTime)
        WHERE RowNum = @CurrentRowNum;
    END

    SET @PrevRowNum = @CurrentRowNum;
    FETCH NEXT FROM row_cursor INTO @CurrentRowNum;
END

CLOSE row_cursor;
DEALLOCATE row_cursor;

-- Calculate EndDateTime = StartDateTime + Hours
UPDATE pd
SET EndDateTime = DATEADD(HOUR, ed.Hours, pd.StartDateTime)
FROM #ProcessedData pd
INNER JOIN #ExcelData ed ON ed.RowNum = pd.RowNum;
-- ================================================================================================
-- STEP 3: VALIDATION - Review data before inserting
-- ================================================================================================

-- Check for missing SectionIds
IF EXISTS (SELECT 1 FROM #ProcessedData WHERE SectionId IS NULL)
BEGIN
    PRINT 'WARNING: Found rows with NULL SectionId (section not found in M2.Sections):'
    SELECT RowNum, StartDateTime, ProductDescription 
    FROM #ProcessedData 
    WHERE SectionId IS NULL
    ORDER BY RowNum
    
    PRINT ''
    PRINT 'Please add missing sections to M2.Sections table first, or remove these rows.'
    PRINT 'Aborting import.'
    RETURN
END

SELECT s.SectionName, pd.* FROM #ProcessedData pd LEFT JOIN M2.Sections s on pd.SectionId = s.SectionId

PRINT ''
PRINT '=== SAMPLE DATA (First 5 rows) ==='
SELECT TOP 5 
    RowNum,
    StartDateTime,
    EndDateTime,
    SectionId,
    StatusCode,
    LEFT(ProductDescription, 30) AS ProductDesc,
    ActualChargedTons,
    PlannedTons
FROM #ProcessedData
ORDER BY RowNum;

-- ================================================================================================
-- STEP 4: EXECUTE UPSERT (UNCOMMENT TO RUN)
-- ================================================================================================

DECLARE @EventId INT;
DECLARE @CurrentRow INT = 1;
DECLARE @TotalRows INT = (SELECT COUNT(*) FROM #ProcessedData);
DECLARE @ErrorCount INT = 0;

PRINT ''
PRINT '=== STARTING IMPORT ==='
PRINT 'Processing ' + CAST(@TotalRows AS VARCHAR(10)) + ' rows...'
PRINT ''
DECLARE @Start DATETIME, @End DATETIME, @SectionId INT, @StatusCode CHAR(1);
    DECLARE @Comments VARCHAR(MAX);
    DECLARE @ActualChargedTons REAL, @ActualRolledTons REAL, @ActualFinishedTons REAL;
    DECLARE @ActualTPH REAL, @ActualRollingHours REAL, @ActualMillYield REAL, @ActualFinishedYield REAL;
    DECLARE @ScheduledChargedTons REAL, @ScheduledRolledTons REAL, @ScheduledFinishedTons REAL, @ScheduledRollingHours REAL, @ScheduledMillYield REAL, @ScheduledFinishedYield REAL;
    DECLARE @ScheduledTPH REAL, @PlannedTons REAL, @PlannedTPH REAL, @PlannedMillYield REAL, @PlannedFinishedYield REAL;
	DECLARE @InquiryStatusCode VARCHAR(2)
	SELECT 
        *
    FROM #ProcessedData pd
	LEFT JOIN M2.Sections s ON s.SectionId = pd.SectionId
WHILE @CurrentRow <= @TotalRows
BEGIN
     
    -- Get data for current row
    SELECT 
        @Start = pd.StartDateTime,
        @End = pd.EndDateTime,
        @SectionId = pd.SectionId,
        @StatusCode = pd.StatusCode,
        @Comments = ProductDescription,
        @ActualChargedTons = ActualChargedTons,
        @ActualRolledTons = ActualRolledTons,
        @ActualFinishedTons = ActualFinishedTons,
        @ActualTPH = ActualTPH,
        @ActualRollingHours = ActualRollingHours,
        @ActualMillYield = ActualMillYield,
		@ActualFinishedYield = ActualFinYield,
        @ScheduledChargedTons = ScheduledChargedTons,
        @ScheduledRolledTons = ScheduledRolledTons,
        @ScheduledFinishedTons = ScheduledFinishedTons,
		@ScheduledMillYield = ScheduledMillYield,
		@ScheduledFinishedYield = ScheduledFinYield,
        @ScheduledTPH = ScheduledTPH,
		@ScheduledRollingHours = @ScheduledRollingHours,
        @PlannedTons = PlannedTons,
        @PlannedTPH = s.TonsPerHour,
        @PlannedMillYield =  s.MillYield,
		@PlannedFinishedYield = s.FinishedYield,
		@InquiryStatusCode= pd.InquiryStatusCode
    FROM #ProcessedData pd
	LEFT JOIN M2.Sections s ON s.SectionId = pd.SectionId
    WHERE RowNum = @CurrentRow;
	
    BEGIN TRY
        -- Call UpsertProductionEvent
        EXEC @EventId = [M2].[UpsertProductionEvent]
            @EventId = NULL,
            @Start = @Start,
            @End = @End,
            @SectionId = @SectionId,
            @StatusCode = @StatusCode,
            @PlannedTons = @PlannedTons,
            @PlannedTPH = @PlannedTPH,
            @PlannedMillYield = @PlannedMillYield,
			@PlannedFinishedYield = @PlannedFinishedYield,
            @ScheduledChargedTons = @ScheduledChargedTons,
            @ScheduledRolledTons = @ScheduledRolledTons,
            @ScheduledFinishedTons = @ScheduledFinishedTons,
            @ScheduledTPH = @ScheduledTPH,
			@ScheduledRollingHours = @ScheduledRollingHours,
			@ScheduledMillYield = @ScheduledMillYield,
			@ScheduledFinishedYield = @ScheduledFinishedYield,
            @ActualChargedTons = @ActualChargedTons,
            @ActualRolledTons = @ActualRolledTons,
            @ActualFinishedTons = @ActualFinishedTons,
            @ActualTPH = @ActualTPH,
            @ActualRollingHours = @ActualRollingHours,
            @ActualMillYield = @ActualMillYield,
			@ActualFinishedYield = @ActualFinishedYield,
			@InquiryStatusCode = @InquiryStatusCode,
            --@Comments = @Comments,
            @User = @User;

        IF @CurrentRow % 10 = 0
            PRINT 'Processed ' + CAST(@CurrentRow AS VARCHAR(10)) + ' of ' + CAST(@TotalRows AS VARCHAR(10)) + ' rows...';

    END TRY
    BEGIN CATCH
        SET @ErrorCount = @ErrorCount + 1;
        PRINT 'ERROR on row ' + CAST(@CurrentRow AS VARCHAR(10)) + ': ' + ERROR_MESSAGE();
    END CATCH

    SET @CurrentRow = @CurrentRow + 1;
END

PRINT ''
PRINT '=== IMPORT COMPLETE ==='
PRINT 'Successfully processed: ' + CAST((@TotalRows - @ErrorCount) AS VARCHAR(10)) + ' rows'
PRINT 'Errors: ' + CAST(@ErrorCount AS VARCHAR(10))

-- ================================================================================================
-- CLEANUP
-- ================================================================================================
 DROP TABLE #ExcelData;
 DROP TABLE #ProcessedData;

-- ================================================================================================
-- Holydays and Downtimes
-- ================================================================================================
-- 2025/09/01 - Holiday

SET @Start = UTL.ToUTC('2025-09-01 00:00:00')
SET @End = UTL.ToUTC('2025-09-02 00:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'H', 
    @Comments = 'Holiday', 
    @User = 'Admin'

-- 2025/09/10 - 16 hr downday
SET @Start = UTL.ToUTC('2025-09-10 00:00:00')
SET @End = UTL.ToUTC('2025-09-10 16:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'D', 
    @Comments = '16 hr downday', 
    @User = 'Admin'

-- 2025/09/24 - 16 hr downday

SET @Start = UTL.ToUTC('2025-09-24 12:00:00')
SET @End = UTL.ToUTC('2025-09-25 04:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'D', 
    @Comments = '16 hr downday', 
    @User = 'Admin'

-- 2025/10/22 - 18 hr downday

SET @Start = UTL.ToUTC('2025-10-22 00:00:00')
SET @End = UTL.ToUTC('2025-10-22 18:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'D', 
    @Comments = '18 hr downday', 
    @User = 'Admin'

-- 2025/10/28 - 15 hr downday

SET @Start = UTL.ToUTC('2025-10-28 21:00:00')
SET @End = UTL.ToUTC('2025-10-29 12:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'D', 
    @Comments = '15 hr downday', 
    @User = 'Admin'

-- 2025/10/30 - 6 hr downday

SET @Start = UTL.ToUTC('2025-10-30 06:00:00')
SET @End = UTL.ToUTC('2025-10-30 12:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'D', 
    @Comments = 'divide shear trial heat. 6hr downday', 
    @User = 'Admin'

-- 2025/11/11 - 12 hr downday

SET @Start = UTL.ToUTC('2025-11-11 00:00:00')
SET @End = UTL.ToUTC('2025-11-11 12:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'D', 
    @Comments = 'planned 12 hr downday', 
    @User = 'Admin'

-- 2025/11/27 - Holiday  

SET @Start = UTL.ToUTC('2025-11-27 00:00:00')
SET @End = UTL.ToUTC('2025-11-28 00:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End, 
    @EventTypeCode = 'H', 
    @Comments = 'Holiday', 
    @User = 'Admin'

-- 2025/11/28--2025/12/1  - Downtime (OUTAGE)

SET @Start = UTL.ToUTC('2025-11-28 00:00:00')
SET @End = UTL.ToUTC('2025-12-02 00:00:00')

EXEC [M2].[UpsertEvent] 
    @Start = @Start, 
    @End = @End,  
    @EventTypeCode = 'D', 
    @Comments = 'OUTAGE', 
    @User = 'Admin'

-- 2025/12/09  - planned 12 hr downday

SET @Start = UTL.ToUTC('2025-12-09 00:00:00')
SET @End = UTL.ToUTC('2025-12-09 12:00:00')

EXEC [M2].[UpsertEvent]
    @Start = @Start,
    @End = @End,
    @EventTypeCode = 'D',
    @Comments = 'planned 12 hr downday',
    @User = 'Admin'

-- 2025/12/11 - planned 12 hr downday

SET @Start = UTL.ToUTC('2025-12-11 00:00:00')
SET @End = UTL.ToUTC('2025-12-11 12:00:00')

EXEC [M2].[UpsertEvent]
    @Start = @Start,
    @End = @End,
    @EventTypeCode = 'D',
    @Comments = 'planned 12 hr downday',
    @User = 'Admin'

-- 2025/12/24 - Holiday (Christmas Eve)

SET @Start = UTL.ToUTC('2025-12-24 00:00:00')
SET @End = UTL.ToUTC('2025-12-25 00:00:00')

EXEC [M2].[UpsertEvent]
    @Start = @Start,
    @End = @End,
    @EventTypeCode = 'H',
    @Comments = 'Holiday',
    @User = 'Admin'

-- 2025/12/25 - Holiday (Christmas Day)

SET @Start = UTL.ToUTC('2025-12-25 00:00:00')
SET @End = UTL.ToUTC('2025-12-26 00:00:00')

EXEC [M2].[UpsertEvent]
    @Start = @Start,
    @End = @End,
    @EventTypeCode = 'H',
    @Comments = 'Holiday',
    @User = 'Admin'

UPDATE M2.Events 
	SET [End] = '2025-11-27 05:00:00.000'
WHERE [Start] = '2025-11-25 17:00:00.000'

UPDATE M2.ProductionEvents
	SET BilletsStatusCode = 'C'
WHERE StatusCode = 'P'

-- ================================================================================================
-- Campaigns Sync
-- ================================================================================================

EXEC [M2].[SyncCampaignsToProductionEvents]

